document.addEventListener('DOMContentLoaded', () => {
  const listElement = document.getElementById('chat-list');
  const searchInput = document.getElementById('search');
  const clearAllBtn = document.getElementById('clear-all');
  const exportBtn = document.getElementById('export-btn');

  let allChats = [];

  loadChats();

  searchInput.addEventListener('input', (e) => {
    const term = e.target.value.toLowerCase();
    const filtered = allChats.filter(chat => chat.title.toLowerCase().includes(term));
    renderList(filtered);
  });

  clearAllBtn.addEventListener('click', () => {
    if (confirm("בטוח שאתה רוצה למחוק את כל ההיסטוריה? פעולה זו לא ניתנת לביטול.")) {
      chrome.storage.local.set({ savedChats: [] }, loadChats);
    }
  });

  exportBtn.addEventListener('click', exportData);

  function loadChats() {
    chrome.storage.local.get({ savedChats: [] }, (result) => {
      allChats = result.savedChats;
      renderList(allChats);
    });
  }

  // פונקציה מסייעת לקביעת קבוצת זמן
  function getDateGroup(timestamp) {
    const date = new Date(timestamp);
    const now = new Date();
    const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());
    const yesterday = new Date(today);
    yesterday.setDate(yesterday.getDate() - 1);
    const lastWeek = new Date(today);
    lastWeek.setDate(lastWeek.getDate() - 7);

    if (date >= today) return "היום";
    if (date >= yesterday) return "אתמול";
    if (date >= lastWeek) return "7 הימים האחרונים";
    return "ישנים יותר";
  }

  function renderList(chats) {
    listElement.innerHTML = '';

    if (chats.length === 0) {
      listElement.innerHTML = `
        <div class="empty-state">
          <svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="#ddd" stroke-width="1">
            <path d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z" stroke-linecap="round" stroke-linejoin="round"/>
          </svg>
          <p>לא נמצאו שיחות</p>
        </div>`;
      return;
    }

    // קיבוץ לפי זמנים
    const groups = {
      "היום": [],
      "אתמול": [],
      "7 הימים האחרונים": [],
      "ישנים יותר": []
    };

    chats.forEach(chat => {
      const group = getDateGroup(chat.timestamp);
      groups[group].push(chat);
    });

    // רינדור לפי סדר הקבוצות
    Object.keys(groups).forEach(groupName => {
      const groupChats = groups[groupName];
      if (groupChats.length > 0) {
        
        // כותרת קבוצה
        const groupHeader = document.createElement('div');
        groupHeader.className = 'time-group-label';
        groupHeader.textContent = groupName;
        listElement.appendChild(groupHeader);

        // רשימת השיחות בקבוצה
        groupChats.forEach(chat => {
          const item = createChatItem(chat);
          listElement.appendChild(item);
        });
      }
    });
  }

  function createChatItem(chat) {
    const li = document.createElement('div');
    li.className = 'chat-item';
    li.onclick = (e) => {
       // אם לחצו על כפתורים או עריכה, לא לפתוח את הלינק
       if (e.target.closest('button') || e.target.closest('input')) return;
       chrome.tabs.create({ url: chat.url });
    };

    // אייקון הודעה
    const iconSvg = `
    <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="#5f6368" stroke-width="2" style="margin-left:10px">
      <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"></path>
    </svg>`;

    const infoDiv = document.createElement('div');
    infoDiv.className = 'chat-info';
    
    // כותרת (ניתנת לעריכה)
    const titleSpan = document.createElement('div');
    titleSpan.className = 'chat-title';
    titleSpan.textContent = chat.title;
    titleSpan.title = chat.title; // Tooltip

    const metaSpan = document.createElement('div');
    metaSpan.className = 'chat-meta';
    // הצגת שעה עבור "היום", תאריך עבור השאר
    const dateObj = new Date(chat.timestamp);
    const timeStr = getDateGroup(chat.timestamp) === "היום" 
      ? dateObj.toLocaleTimeString('he-IL', {hour: '2-digit', minute:'2-digit'})
      : dateObj.toLocaleDateString('he-IL');
    metaSpan.textContent = timeStr;

    infoDiv.innerHTML = iconSvg;
    infoDiv.appendChild(titleSpan);
    infoDiv.appendChild(metaSpan);

    // פעולות
    const actionsDiv = document.createElement('div');
    actionsDiv.className = 'actions';

    // כפתור עריכה
    const editBtn = document.createElement('button');
    editBtn.className = 'btn-icon';
    editBtn.innerHTML = `<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"></path><path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"></path></svg>`;
    editBtn.title = "ערוך שם";
    editBtn.onclick = (e) => {
      e.stopPropagation();
      startEditing(chat, titleSpan, infoDiv);
    };

    // כפתור מחיקה
    const deleteBtn = document.createElement('button');
    deleteBtn.className = 'btn-icon delete';
    deleteBtn.innerHTML = `<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><line x1="18" y1="6" x2="6" y2="18"></line><line x1="6" y1="6" x2="18" y2="18"></line></svg>`;
    deleteBtn.title = "מחק";
    deleteBtn.onclick = (e) => {
      e.stopPropagation();
      deleteChat(chat.url);
    };

    actionsDiv.appendChild(editBtn);
    actionsDiv.appendChild(deleteBtn);

    // צריך לסדר את המבנה הפנימי של ה-infoDiv כי הוספנו לו SVG ב-innerHTML
    // דרך נקייה יותר להרכבה:
    infoDiv.innerHTML = ''; 
    const iconContainer = document.createElement('span');
    iconContainer.innerHTML = iconSvg;
    
    const textContainer = document.createElement('div');
    textContainer.style.display = 'inline-block';
    textContainer.style.verticalAlign = 'middle';
    textContainer.appendChild(titleSpan);
    textContainer.appendChild(metaSpan);

    infoDiv.appendChild(iconContainer);
    infoDiv.appendChild(textContainer);


    li.appendChild(infoDiv);
    li.appendChild(actionsDiv);
    
    return li;
  }

  function startEditing(chat, titleEl, container) {
    const input = document.createElement('input');
    input.type = 'text';
    input.value = chat.title;
    input.className = 'chat-title editing';
    input.style.width = '100%';
    
    // החלפת האלמנט טקסט באלמנט אינפוט
    titleEl.replaceWith(input);
    input.focus();

    const saveEdit = () => {
      const newTitle = input.value.trim();
      if (newTitle && newTitle !== chat.title) {
        updateChatTitleInStorage(chat.url, newTitle);
      } else {
        // ביטול עריכה
        input.replaceWith(titleEl);
      }
    };

    input.addEventListener('blur', saveEdit);
    input.addEventListener('keydown', (e) => {
      if (e.key === 'Enter') {
        input.blur();
      }
    });
    input.onclick = (e) => e.stopPropagation();
  }

  function updateChatTitleInStorage(url, newTitle) {
    const chatIndex = allChats.findIndex(c => c.url === url);
    if (chatIndex > -1) {
      allChats[chatIndex].title = newTitle;
      allChats[chatIndex].isEdited = true; // מסמנים שנערך ידנית כדי שעדכון אוטומטי לא ידרוס
      chrome.storage.local.set({ savedChats: allChats }, loadChats);
    }
  }

  function deleteChat(url) {
    allChats = allChats.filter(chat => chat.url !== url);
    chrome.storage.local.set({ savedChats: allChats }, loadChats);
  }

  function exportData() {
    const dataStr = "data:text/json;charset=utf-8," + encodeURIComponent(JSON.stringify(allChats, null, 2));
    const downloadAnchorNode = document.createElement('a');
    downloadAnchorNode.setAttribute("href", dataStr);
    downloadAnchorNode.setAttribute("download", "gemini_history.json");
    document.body.appendChild(downloadAnchorNode);
    downloadAnchorNode.click();
    downloadAnchorNode.remove();
  }
});